<?php namespace Laravel\Security;

class Crypter {

	/**
	 * The encryption cipher.
	 *
	 * @var string
	 */
	public $cipher;

	/**
	 * The encryption mode.
	 *
	 * @var string
	 */
	public $mode;

	/**
	 * The encryption key.
	 *
	 * @var string
	 */
	public $key;

	/**
	 * Create a new Crypter instance.
	 *
	 * A valid cipher and mode supported by the Mcrypt extension must be given to the constructor.
	 * Also, an encryption key (typically from the application configuration) must be specified.
	 *
	 * @param  string  $cipher
	 * @param  string  $mode
	 * @param  string  $key
	 * @return void
	 */
	public function __construct($cipher, $mode, $key)
	{
		$this->key = $key;
		$this->mode = $mode;
		$this->cipher = $cipher;

		if (trim((string) $this->key) === '')
		{
			throw new \Exception('The encryption class may not be used without an encryption key.');
		}
	}

	/**
	 * Encrypt a string using Mcrypt.
	 *
	 * The string will be encrypted using the cipher and mode specified when the crypter
	 * instance was created, and the final result will be base64 encoded.
	 *
	 * <code>
	 *		// Encrypt a string using the Mcrypt PHP extension
	 *		$encrypted = Crypter::encrpt('secret');
	 * </code>
	 *
	 * @param  string  $value
	 * @return string
	 */
	public function encrypt($value)
	{
		// Determine the most appropriate random number generator for the operating
		// system and environment the application is running on.
		if (defined('MCRYPT_DEV_URANDOM'))
		{
			$randomizer = MCRYPT_DEV_URANDOM;
		}
		elseif (defined('MCRYPT_DEV_RANDOM'))
		{
			$randomizer = MCRYPT_DEV_RANDOM;
		}
		else
		{
			$randomizer = MCRYPT_RAND;			
		}

		$iv = mcrypt_create_iv($this->iv_size(), $randomizer);

		return base64_encode($iv.mcrypt_encrypt($this->cipher, $this->key, $value, $this->mode, $iv));
	}

	/**
	 * Decrypt a string using Mcrypt.
	 *
	 * The string will be decrypted using the cipher and mode specified when the crypter was created.
	 *
	 * <code>
	 *		// Decrypt a string using the Mcrypt PHP extension
	 *		$decrypted = Crypter::decrypt($secret);
	 * </code>
	 *
	 * @param  string  $value
	 * @return string
	 */
	public function decrypt($value)
	{
		// Since all encrypted strings generated by this class are base64 encoded, we will
		// first attempt to base64 decode the string. If we can't do it, we'll bail out.
		if ( ! is_string($value = base64_decode($value, true)))
		{
			throw new \Exception('Decryption error. Input value is not valid base64 data.');
		}

		// Extract the input vector and the encrypted string from the value
		list($iv, $value) = array(substr($value, 0, $this->iv_size()), substr($value, $this->iv_size()));

		return rtrim(mcrypt_decrypt($this->cipher, $this->key, $value, $this->mode, $iv), "\0");
	}

	/**
	 * Get the input vector size for the cipher and mode.
	 *
	 * Different ciphers and modes use varying lengths of input vectors.
	 *
	 * @return int
	 */
	private function iv_size()
	{
		return mcrypt_get_iv_size($this->cipher, $this->mode);
	}

}