<?php namespace Laravel; use Closure;

class Lang {

	/**
	 * All of the loaded language lines.
	 *
	 * The array is keyed by [$language.$file].
	 *
	 * @var array
	 */
	protected static $lines = array();

	/**
	 * The key of the language line being retrieved.
	 *
	 * @var string
	 */
	protected $key;

	/**
	 * The replacements that should be made on the language line.
	 *
	 * @var array
	 */
	protected $replacements;

	/**
	 * The language in which the line should be retrieved.
	 *
	 * @var string
	 */
	protected $language;

	/**
	 * The paths containing the language files.
	 *
	 * @var array
	 */
	protected $paths;

	/**
	 * Create a new Lang instance.
	 *
	 * @param  string  $key
	 * @param  array   $replacements
	 * @param  string  $language
	 * @param  array   $paths
	 * @return void
	 */
	protected function __construct($key, $replacements = array(), $language = null, $paths = array())
	{
		$this->key = $key;
		$this->paths = $paths;
		$this->language = $language;
		$this->replacements = $replacements;
	}

	/**
	 * Create a new language line instance.
	 *
	 * <code>
	 *		// Create a new language line instance for a given line
	 *		$line = Lang::line('validation.required');
	 *
	 *		// Specify some replacements for the language line
	 *		$line = Lang::line('validation.required', array('attribute' => 'email'));
	 * </code>
	 *
	 * @param  string  $key
	 * @param  array   $replacements
	 * @param  string  $language
	 * @param  array   $paths
	 * @return Lang
	 */
	public static function line($key, $replacements = array(), $language = null, $paths = array())
	{
		if (count($paths) == 0) $paths = array(SYS_LANG_PATH, LANG_PATH);

		return new static($key, $replacements, $language, $paths);
	}

	/**
	 * Get the language line as a string.
	 *
	 * If a language is specified, it should correspond to a directory within
	 * your application language directory.
	 *
	 * <code>
	 *		// Get a language line
	 *		$line = Lang::line('validation.required')->get();
	 *
	 *		// Get a language line in a specified language
	 *		$line = Lang::line('validation.required')->get('sp');
	 *
	 *		// Return a default value if the line doesn't exist
	 *		$line = Lang::line('validation.required', null, 'Default');
	 * </code>
	 *
	 * @param  string  $language
	 * @param  string  $default
	 * @return string
	 */
	public function get($language = null, $default = null)
	{
		// If a language was passed to the method, we will let it override the default
		if ( ! is_null($language)) $this->language = $language;

		list($file, $line) = $this->parse($this->key);

		if ( ! $this->load($file))
		{
			return ($default instanceof Closure) ? call_user_func($default) : $default;
		}

		$line = Arr::get(static::$lines[$this->language.$file], $line, $default);

		foreach ($this->replacements as $key => $value)
		{
			$line = str_replace(':'.$key, $value, $line);
		}

		return $line;
	}

	/**
	 * Parse a language key into its file and line segments.
	 *
	 * @param  string  $key
	 * @return array
	 */
	protected function parse($key)
	{
		if (count($segments = explode('.', $key)) > 1)
		{
			return array($segments[0], implode('.', array_slice($segments, 1)));
		}

		throw new \Exception("Invalid language line [$key]. A specific line must be specified.");
	}

	/**
	 * Load all of the language lines from a language file.
	 *
	 * @param  string  $file
	 * @return bool
	 */
	protected function load($file)
	{
		if (isset(static::$lines[$this->language.$file])) return;

		$language = array();

		// Language files cascade. Typically, the system language array is loaded first,
		// followed by the application array. This allows the convenient overriding of the
		// system language files (like validation) by the application developer.
		foreach ($this->paths as $directory)
		{
			if (file_exists($path = $directory.$this->language.'/'.$file.EXT))
			{
				$language = array_merge($language, require $path);
			}
		}

		// If language lines were actually found, they will be loaded into the array
		// containing all of the lines for all languages and files. The array is
		// keyed by the language and the file name.
		if (count($language) > 0) static::$lines[$this->language.$file] = $language;
		
		return isset(static::$lines[$this->language.$file]);		
	}

	/**
	 * Get the string content of the language line.
	 */
	public function __toString() { return $this->get(); }

}